<?php
defined( 'ABSPATH' ) || exit;

class HRM_Attendance {

    public static function init() {
        add_action( 'wp_ajax_hrm_mark_attendance', [ __CLASS__, 'handle_ajax_mark_attendance' ] );
        add_action( 'wp_ajax_hrm_edit_attendance', [ __CLASS__, 'handle_ajax_edit_attendance' ] );
        add_action( 'wp_ajax_hrm_manual_attendance', [ __CLASS__, 'handle_ajax_manual_attendance' ] );

        add_action( 'wp_ajax_hrm_delete_attendance', [ __CLASS__, 'handle_ajax_delete_attendance' ] );
        add_action( 'wp_ajax_hrm_export_attendance', [ __CLASS__, 'handle_ajax_export_attendance' ] );
        add_action( 'wp_ajax_hrm_bulk_attendance', [ __CLASS__, 'handle_ajax_bulk_attendance' ] );
        add_action( 'hrm_daily_absent_check', [ __CLASS__, 'hrm_check_absentees' ] );
    }

    public static function handle_ajax_delete_attendance() {
        check_admin_referer( 'hrm_ajax_nonce', 'nonce' );

        if ( ! current_user_can( 'manage_options' ) ) {
            wp_send_json_error( 'Unauthorized' );
        }

        global $wpdb;
        $ids = isset($_POST['ids']) ? $_POST['ids'] : [];
        
        if ( empty($ids) || !is_array($ids) ) {
            wp_send_json_error( 'No records selected.' );
        }

        $ids_sanitized = array_map( 'intval', $ids );
        $ids_string = implode(',', $ids_sanitized);

        // Fetch records to process
        $records_to_process = $wpdb->get_results( "SELECT id, employee_id, date FROM {$wpdb->prefix}hrm_attendance WHERE id IN ($ids_string)" );
        
        $deleted_count = 0;
        $skipped_count = 0;
        
        foreach ($records_to_process as $record) {
             // Check total count for this emp/date
             $total_count = $wpdb->get_var( $wpdb->prepare(
                 "SELECT COUNT(*) FROM {$wpdb->prefix}hrm_attendance WHERE employee_id = %d AND date = %s",
                 $record->employee_id, $record->date
             ));

             // Only delete if there is more than 1 record (duplicate)
             if ( $total_count > 1 ) {
                 $wpdb->delete( $wpdb->prefix . 'hrm_attendance', ['id' => $record->id] );
                 $deleted_count++;
             } else {
                 $skipped_count++;
             }
        }
        
        if ($deleted_count === 0 && $skipped_count > 0) {
            wp_send_json_error('Cannot delete the only record for the employee/day.');
        }

        wp_send_json_success( "$deleted_count record(s) deleted." . ($skipped_count > 0 ? " ($skipped_count kept)" : "") );
    }

    public static function handle_ajax_manual_attendance() {
        check_admin_referer( 'hrm_ajax_nonce', 'nonce' );
        if ( ! current_user_can( 'manage_options' ) ) wp_send_json_error( 'Unauthorized' );
        
        global $wpdb;
        $employee_id = intval( $_POST['employee_id'] );
        $date = sanitize_text_field( $_POST['date'] );
        $check_in = sanitize_text_field( $_POST['check_in'] );
        $check_out = sanitize_text_field( $_POST['check_out'] );
        
        $check_in_dt = $check_in ? date('Y-m-d H:i:s', strtotime("$date $check_in")) : null;
        $check_out_dt = $check_out ? date('Y-m-d H:i:s', strtotime("$date $check_out")) : null;
        $remarks = isset($_POST['remarks']) ? sanitize_textarea_field($_POST['remarks']) : '';
        
        $hours = 0;
        if ( $check_in_dt && $check_out_dt ) {
            $diff = strtotime($check_out_dt) - strtotime($check_in_dt);
            $hours = round($diff / 3600, 2);
        }

        // Determine status
        $status = 'present'; 
        if ( ! $check_in_dt ) $status = 'absent';
        
        // Upsert
        $exists = $wpdb->get_var( $wpdb->prepare( "SELECT id FROM {$wpdb->prefix}hrm_attendance WHERE employee_id = %d AND date = %s", $employee_id, $date ) );

        if ( $exists ) {
            $wpdb->update( $wpdb->prefix . 'hrm_attendance', 
                ['check_in' => $check_in_dt, 'check_out' => $check_out_dt, 'hours_worked' => $hours, 'status' => $status, 'remarks' => $remarks],
                ['id' => $exists]
            );
        } else {
            $wpdb->insert( $wpdb->prefix . 'hrm_attendance', [
                'employee_id' => $employee_id, 'date' => $date, 'check_in' => $check_in_dt, 'check_out' => $check_out_dt, 'hours_worked' => $hours, 'status' => $status, 'remarks' => $remarks
            ]);
        }
        
        wp_send_json_success( 'Record added successfully.' );
    }

    public static function handle_ajax_bulk_attendance() {
        check_admin_referer( 'hrm_ajax_nonce', 'nonce' );
        if ( ! current_user_can( 'manage_options' ) ) wp_send_json_error( 'Unauthorized' );
        
        global $wpdb;
        $employee_ids = isset($_POST['employee_ids']) ? (array)$_POST['employee_ids'] : [];
        $start_date = sanitize_text_field( $_POST['start_date'] );
        $end_date = sanitize_text_field( $_POST['end_date'] );
        $status = sanitize_key( $_POST['status'] );
        $remarks = isset($_POST['bulk_remarks']) ? sanitize_textarea_field($_POST['bulk_remarks']) : '';
        
        if ( empty($employee_ids) ) wp_send_json_error( 'Please select at least one employee.' );
        if ( empty($start_date) || empty($end_date) ) wp_send_json_error( 'Please select a date range.' );
        
        try {
            $begin = new DateTime($start_date);
            $end = new DateTime($end_date);
            $end->modify('+1 day');
            
            $interval = new DateInterval('P1D');
            $daterange = new DatePeriod($begin, $interval, $end);
            
            $count = 0;
            foreach ($employee_ids as $emp_id) {
                $emp_id = intval($emp_id);
                foreach ($daterange as $date) {
                    $date_str = $date->format("Y-m-d");
                    
                    // Default values based on status
                    $hours = 0;
                    $check_in = null;
                    $check_out = null;

                    if ( in_array($status, ['present', 'late', 'half_day']) ) {
                        // Assuming 8 hours for full day, 4 for half
                        $hours = ($status === 'half_day') ? 4 : 8;
                        
                        // Set dummy check-in/out times for reporting if they are present/late/half_day
                        if ($status === 'half_day') {
                            $check_in = $date_str . ' 11:00:00';
                            $check_out = $date_str . ' 15:00:00';
                        } elseif ($status === 'late') {
                            $check_in = $date_str . ' 11:45:00'; // After 11:15 cutoff
                            $check_out = $date_str . ' 19:00:00';
                        } else {
                            $check_in = $date_str . ' 11:00:00';
                            $check_out = $date_str . ' 19:00:00';
                        }
                    }

                    $exists = $wpdb->get_var( $wpdb->prepare( "SELECT id FROM {$wpdb->prefix}hrm_attendance WHERE employee_id = %d AND date = %s", $emp_id, $date_str ) );
                    
                    if ( $exists ) {
                        $wpdb->update( $wpdb->prefix . 'hrm_attendance', 
                            [
                                'status' => $status, 
                                'remarks' => $remarks, 
                                'hours_worked' => $hours,
                                'check_in' => $check_in,
                                'check_out' => $check_out,
                                'updated_at' => current_time('mysql')
                            ],
                            ['id' => $exists]
                        );
                    } else {
                        $wpdb->insert( $wpdb->prefix . 'hrm_attendance', [
                            'employee_id' => $emp_id, 
                            'date' => $date_str, 
                            'status' => $status, 
                            'remarks' => $remarks,
                            'hours_worked' => $hours,
                            'check_in' => $check_in,
                            'check_out' => $check_out,
                            'created_at' => current_time('mysql')
                        ]);
                    }
                    $count++;
                }
            }
            
            wp_send_json_success( "Successfully processed attendance for " . count($employee_ids) . " employee(s) across " . count(iterator_to_array($daterange)) . " day(s)." );
        } catch ( Exception $e ) {
            wp_send_json_error( "Error processing date range: " . $e->getMessage() );
        }
    }

    public static function handle_ajax_edit_attendance() {
        check_admin_referer( 'hrm_ajax_nonce', 'nonce' );

        if ( ! current_user_can( 'manage_options' ) ) {
            wp_send_json_error( 'Unauthorized' );
        }

        global $wpdb;
        $record_id = intval( $_POST['record_id'] );
        $date = sanitize_text_field( $_POST['date'] );
        $check_in = sanitize_text_field( $_POST['check_in'] ); // Format: H:i
        $check_out = sanitize_text_field( $_POST['check_out'] ); // Format: H:i
        $status = sanitize_key( $_POST['status'] );
        $remarks = isset($_POST['remarks']) ? sanitize_textarea_field($_POST['remarks']) : '';
        
        // Convert times to full datetime
        $check_in_dt = $check_in ? date('Y-m-d H:i:s', strtotime("$date $check_in")) : null;
        $check_out_dt = $check_out ? date('Y-m-d H:i:s', strtotime("$date $check_out")) : null;

        // Recalculate hours
        $hours = 0;
        if ( $check_in_dt && $check_out_dt ) {
            $diff = strtotime($check_out_dt) - strtotime($check_in_dt);
            $hours = round($diff / 3600, 2);
        }

        $result = $wpdb->update( 
            $wpdb->prefix . 'hrm_attendance',
            [
                'date' => $date,
                'check_in' => $check_in_dt,
                'check_out' => $check_out_dt,
                'hours_worked' => $hours,
                'status' => $status,
                'remarks' => $remarks
            ],
            [ 'id' => $record_id ]
        );

        if ( $result !== false ) {
            HRM_Security::log( 'attendance_edit', "Attendance record #$record_id updated by Admin." );
            wp_send_json_success( 'Record updated successfully.' );
        } else {
            wp_send_json_error( 'Failed to update record.' );
        }
    }

    public static function hrm_check_absentees() {
        global $wpdb;
        $today = current_time( 'Y-m-d' );
        $yesterday = date('Y-m-d', strtotime('-1 day', strtotime($today)));

        // 1. Mark 'Incomplete' for yesterday's records with no check-out
        $wpdb->query( $wpdb->prepare(
            "UPDATE {$wpdb->prefix}hrm_attendance 
             SET status = 'incomplete' 
             WHERE date = %s AND check_in IS NOT NULL AND check_out IS NULL AND status != 'incomplete'",
            $yesterday
        ));

        // 2. Mark 'Absent' for yesterday (safer than today)
        // Check for employees with NO record for yesterday
        $employees = $wpdb->get_col( $wpdb->prepare( 
            "SELECT ID FROM {$wpdb->posts} WHERE post_type = %s AND post_status = 'publish'", 
            HRM_Setup::EMPLOYEE_POST_TYPE 
        ));

        if ( ! $employees ) return;

        foreach ( $employees as $emp_id ) {
            $exists = $wpdb->get_var( $wpdb->prepare(
                "SELECT id FROM {$wpdb->prefix}hrm_attendance WHERE employee_id = %d AND date = %s",
                $emp_id, $yesterday
            ));

            if ( ! $exists ) {
                // Determine if yesterday was a working day? Assuming yes for now.
                $wpdb->insert( $wpdb->prefix . 'hrm_attendance', [
                    'employee_id' => $emp_id,
                    'date' => $yesterday,
                    'check_in' => null,
                    'check_out' => null,
                    'status' => 'absent',
                    'hours_worked' => 0
                ]);
            }
        }
    }

    public static function handle_ajax_mark_attendance() {
        // Suppress display errors to ensure clean JSON
        ini_set('display_errors', 0);
        error_reporting(0);
        
        $debug_log = HRM_PLUGIN_PATH . 'hrm_debug.log';
        
        // Comprehensive Debug Logging
        $debug_info = [
            'time' => current_time('mysql'),
            'method' => $_SERVER['REQUEST_METHOD'],
            'post_data' => $_POST,
            'user_logged_in' => is_user_logged_in(),
            'user_id' => get_current_user_id()
        ];
        
        error_log( "[" . date('Y-m-d H:i:s') . "] HRM AJAX Request: " . json_encode($debug_info) . "\n", 3, $debug_log );
        
        // Graceful Nonce Check (don't die silently)
        $nonce = isset($_POST['nonce']) ? sanitize_text_field($_POST['nonce']) : '';
        if ( empty($nonce) || !wp_verify_nonce($nonce, 'hrm_ajax_nonce') ) {
            error_log( "[" . date('Y-m-d H:i:s') . "] HRM Nonce Failed - Received: '$nonce'\n", 3, $debug_log );
            wp_send_json_error('Security verification failed. Please refresh the page and try again.');
            exit;
        }
        
        ob_start();

        try {
            if ( ! is_user_logged_in() ) {
                error_log( "[" . date('Y-m-d H:i:s') . "] HRM Error: User not logged in\n", 3, $debug_log );
                throw new \Exception( 'User not logged in. Please log in and try again.' );
            }

            global $wpdb;
            $current_user = wp_get_current_user();
            $user_ip = $_SERVER['REMOTE_ADDR'];
            if ( $user_ip === '::1' ) $user_ip = '127.0.0.1';
            
            // --- IP Match Logic (Using new IP Configuration module) ---
            if ( class_exists( 'HRM_IP_Config' ) && ! HRM_IP_Config::is_ip_allowed( $user_ip ) ) {
                // IP not in whitelist and whitelist has active entries
                $active_ips = HRM_IP_Config::get_active_ips();
                if ( ! empty( $active_ips ) ) {
                    HRM_Security::log( 'access_denied', "Attendance check-in blocked: IP $user_ip not in allowed list." );
                    throw new \Exception( 'You are not connected to an authorized network. (' . $user_ip . ')' );
                }
            }

            // --- Employee ID ---
            $post_type = class_exists('HRM_Setup') ? HRM_Setup::EMPLOYEE_POST_TYPE : 'hrm_employee';
            
            // Debug: Log employee lookup
            error_log( "[" . date('Y-m-d H:i:s') . "] HRM Looking for employee - User ID: {$current_user->ID}, Post Type: $post_type\n", 3, $debug_log );
            
            $employee_id = $wpdb->get_var( $wpdb->prepare( 
                "SELECT ID FROM {$wpdb->posts} WHERE post_author = %d AND post_type = %s AND post_status = 'publish' LIMIT 1", 
                $current_user->ID, $post_type 
            ));
            
            // Debug: Log result
            error_log( "[" . date('Y-m-d H:i:s') . "] HRM Employee ID Result: " . ($employee_id ? $employee_id : 'NOT FOUND') . "\n", 3, $debug_log );
            
            if ( ! $employee_id ) {
                // Try fallback: search by email
                $employee_by_email = $wpdb->get_var( $wpdb->prepare(
                    "SELECT p.ID FROM {$wpdb->posts} p 
                     INNER JOIN {$wpdb->postmeta} pm ON p.ID = pm.post_id 
                     WHERE p.post_type = %s AND p.post_status = 'publish' 
                     AND pm.meta_key = 'hrm_email' AND pm.meta_value = %s LIMIT 1",
                    $post_type, $current_user->user_email
                ));
                
                if ( $employee_by_email ) {
                    $employee_id = $employee_by_email;
                    error_log( "[" . date('Y-m-d H:i:s') . "] HRM Employee found by email fallback: $employee_id\n", 3, $debug_log );
                } else {
                    error_log( "[" . date('Y-m-d H:i:s') . "] HRM Error: No employee record for user {$current_user->ID} ({$current_user->user_email})\n", 3, $debug_log );
                    throw new \Exception( 'Employee record not found. Please contact HR to link your account.' );
                }
            }

            $type = isset( $_POST['type'] ) ? sanitize_key( $_POST['type'] ) : '';
            
            if ( empty($type) ) {
                error_log( "[" . date('Y-m-d H:i:s') . "] HRM Error: Missing 'type' parameter\n", 3, $debug_log );
                throw new \Exception( 'Invalid request: missing attendance type.' );
            }
            
            $today = current_time( 'Y-m-d' );
            $now = current_time( 'mysql' );
            $now_ts = time(); // Use standard Unix timestamp (UTC) for JS timer
            
            // Office Logic: 11:00 AM Start | Late after 11:15 AM
            $cutoff_time_str = $today . ' 11:15:00';
            $cutoff_ts = strtotime($cutoff_time_str);

            if ( $type === 'check_in' ) {
                error_log( "[" . date('Y-m-d H:i:s') . "] HRM Check-In: Employee $employee_id for date $today\n", 3, $debug_log );
                
                // Get ALL records for today
                $all_records = $wpdb->get_results( $wpdb->prepare(
                    "SELECT id, check_in FROM {$wpdb->prefix}hrm_attendance WHERE employee_id = %d AND date = %s ORDER BY id ASC",
                    $employee_id, $today
                ));
                
                error_log( "[" . date('Y-m-d H:i:s') . "] HRM Check-In: Found " . count($all_records) . " existing record(s)\n", 3, $debug_log );

                $existing = null;
                if ( ! empty( $all_records ) ) {
                    $existing = $all_records[0];
                    error_log( "[" . date('Y-m-d H:i:s') . "] HRM Check-In: Existing record ID={$existing->id}, check_in=" . ($existing->check_in ?: 'NULL') . "\n", 3, $debug_log );
                    
                    // Clean duplicates
                    if ( count( $all_records ) > 1 ) {
                        $ids_to_delete = [];
                        for ( $i = 1; $i < count( $all_records ); $i++ ) {
                            $ids_to_delete[] = $all_records[$i]->id;
                        }
                        if ( ! empty( $ids_to_delete ) ) {
                            $ids_str = implode(',', array_map('intval', $ids_to_delete));
                            $wpdb->query( "DELETE FROM {$wpdb->prefix}hrm_attendance WHERE id IN ($ids_str)" );
                            error_log( "[" . date('Y-m-d H:i:s') . "] HRM Check-In: Cleaned up " . count($ids_to_delete) . " duplicate(s)\n", 3, $debug_log );
                        }
                    }
                } else {
                    error_log( "[" . date('Y-m-d H:i:s') . "] HRM Check-In: No existing records found - this is a first-time check-in\n", 3, $debug_log );
                }

                if ( $existing && ! empty( $existing->check_in ) ) {
                    error_log( "[" . date('Y-m-d H:i:s') . "] HRM Check-In ERROR: Already checked in at {$existing->check_in}\n", 3, $debug_log );
                    throw new \Exception( 'You have already checked in today.' );
                }

                // Determine Status
                $status = ($now_ts > $cutoff_ts) ? 'late' : 'present';
                error_log( "[" . date('Y-m-d H:i:s') . "] HRM Check-In: Status will be '$status' (cutoff: $cutoff_time_str)\n", 3, $debug_log );

                // Upsert
                if ( $existing ) {
                    error_log( "[" . date('Y-m-d H:i:s') . "] HRM Check-In: UPDATING existing record ID={$existing->id}\n", 3, $debug_log );
                    $result = $wpdb->update( $wpdb->prefix . 'hrm_attendance', [
                        'check_in'    => $now,
                        'status'      => $status
                    ], [ 'id' => $existing->id ] );
                    
                    if ( $result === false ) {
                        error_log( "[" . date('Y-m-d H:i:s') . "] HRM Check-In DB ERROR (update): " . $wpdb->last_error . "\n", 3, $debug_log );
                        throw new \Exception( 'Database error during update. Please try again.' );
                    }
                    error_log( "[" . date('Y-m-d H:i:s') . "] HRM Check-In: Update successful, affected rows: $result\n", 3, $debug_log );
                } else {
                    error_log( "[" . date('Y-m-d H:i:s') . "] HRM Check-In: INSERTING new record for employee $employee_id\n", 3, $debug_log );
                    $result = $wpdb->insert( $wpdb->prefix . 'hrm_attendance', [
                        'employee_id' => $employee_id,
                        'date'        => $today,
                        'check_in'    => $now,
                        'check_out'   => null,
                        'hours_worked'=> 0,
                        'status'      => $status
                    ]);
                    
                    if ( $result === false ) {
                        error_log( "[" . date('Y-m-d H:i:s') . "] HRM Check-In DB ERROR (insert): " . $wpdb->last_error . "\n", 3, $debug_log );
                        throw new \Exception( 'Database error during insert. Please try again.' );
                    }
                    $new_id = $wpdb->insert_id;
                    error_log( "[" . date('Y-m-d H:i:s') . "] HRM Check-In: Insert successful, new ID: $new_id\n", 3, $debug_log );
                }

                error_log( "[" . date('Y-m-d H:i:s') . "] HRM Check-In: SUCCESS - Sending JSON response\n", 3, $debug_log );
                
                ob_clean(); 
                wp_send_json_success([
                    'type' => 'check_in',
                    'message' => 'Successfully Checked In (' . ucfirst($status) . ')',
                    'timestamp' => $now_ts, // Unix timestamp for JS timer
                    'check_in_timestamp' => $now_ts, // Explicit check-in time
                    'status' => $status,
                    'formatted_time' => date( 'h:i A', $now_ts ),
                    'check_in_time' => $now,
                    'server_time' => current_time('mysql')
                ]);
                exit; // Prevent fall-through to "Invalid request type" error

            } elseif ( $type === 'check_out' ) {
                $record = $wpdb->get_row( $wpdb->prepare(
                    "SELECT * FROM {$wpdb->prefix}hrm_attendance WHERE employee_id = %d AND date = %s",
                    $employee_id, $today
                ));

                if ( ! $record || empty( $record->check_in ) ) {
                    throw new \Exception( 'You must check in first.' );
                }

                // Check Daily Progress Submission
                if ( class_exists( 'HRM_Tasks' ) && ! HRM_Tasks::has_submitted_progress( $employee_id, $today ) ) {
                    throw new \Exception( 'Please submit your Daily Progress Report before checking out.' );
                }

                $start = strtotime( get_gmt_from_date( $record->check_in ) );
                $diff = time() - $start;
                $actual_hours = round( $diff / 3600, 2 );

                // Maintain existing status unless it's invalid
                $current_status = $record->status;
                if ( ! in_array( $current_status, ['present', 'late'] ) ) {
                      $check_in_ts_record = strtotime($record->check_in);
                      $current_status = ($check_in_ts_record > $cutoff_ts) ? 'late' : 'present';
                }

                $updated = $wpdb->update( $wpdb->prefix . 'hrm_attendance', 
                    [ 
                        'check_out'    => $now, 
                        'hours_worked' => $actual_hours,
                        'status'       => $current_status 
                    ],
                    [ 'id' => $record->id ]
                );

                if ( $updated === false ) {
                    throw new \Exception( 'Database error: Could not update attendance.' );
                }

                if ( class_exists( 'HRM_Tasks' ) ) {
                    HRM_Tasks::send_daily_progress_email( $employee_id, $today );
                }

                ob_clean(); 
                wp_send_json_success([
                    'type' => 'check_out',
                    'message' => 'Checked Out Successfully.',
                    'timestamp' => $now_ts,
                    'status' => $current_status,
                    'formatted_time' => date( 'h:i A', $now_ts ),
                    'hours_worked' => $actual_hours
                ]);
                exit; // Prevent fall-through to "Invalid request type" error
            }

            throw new \Exception( 'Invalid request type.' );

        } catch ( \Throwable $e ) {
            ob_clean(); 
            error_log( date('[Y-m-d H:i:s] ') . "Attendance Error: " . $e->getMessage() . "\n", 3, $debug_log );
            wp_send_json_error( $e->getMessage() );
        }
    }

    public static function handle_ajax_export_attendance() {
        check_admin_referer( 'hrm_ajax_nonce', 'nonce' );
        if ( ! current_user_can( 'manage_options' ) ) wp_die('Unauthorized');
        
        global $wpdb;
        $date = isset($_GET['date']) ? sanitize_text_field($_GET['date']) : date('Y-m-d');
        $month_start = date('Y-m-01', strtotime($date));
        $month_end = date('Y-m-t', strtotime($date));
        
        $records = $wpdb->get_results( $wpdb->prepare(
            "SELECT * FROM {$wpdb->prefix}hrm_attendance WHERE date BETWEEN %s AND %s ORDER BY date DESC, created_at DESC",
            $month_start, $month_end
        ));
        
        header('Content-Type: text/csv');
        header('Content-Disposition: attachment; filename="attendance_report_'.date('M_Y', strtotime($date)).'.csv"');
        
        $output = fopen('php://output', 'w');
        fputcsv($output, ['Employee Name', 'Date', 'Check In', 'Check Out', 'Hours', 'Status', 'Remarks']);
        
        foreach ($records as $row) {
            $emp_post = get_post($row->employee_id);
            $name = $emp_post ? $emp_post->post_title : 'Unknown (ID:'.$row->employee_id.')';
            
            fputcsv($output, [
                $name,
                $row->date,
                $row->check_in,
                $row->check_out,
                $row->hours_worked,
                $row->status,
                isset($row->remarks) ? $row->remarks : ''
            ]);
        }
        fclose($output);
        exit;
    }

    public static function render_attendance() {
        global $wpdb;
        $table = $wpdb->prefix . 'hrm_attendance';
        
        $today = current_time('Y-m-d');
        $date_filter = isset($_GET['date']) ? sanitize_text_field($_GET['date']) : $today;
        
        $total_today = $wpdb->get_var( $wpdb->prepare( "SELECT COUNT(*) FROM $table WHERE date = %s", $today ));
        $present_today = $wpdb->get_var( $wpdb->prepare( "SELECT COUNT(*) FROM $table WHERE date = %s AND status = 'present'", $today ));
        $absent_today = $wpdb->get_var( $wpdb->prepare( "SELECT COUNT(*) FROM $table WHERE date = %s AND status = 'absent'", $today ));
        $late_today = $wpdb->get_var( $wpdb->prepare( "SELECT COUNT(*) FROM $table WHERE date = %s AND status = 'late'", $today ));

        // Pagination
        $per_page = HRM_Pagination::get_per_page();
        $current_page = HRM_Pagination::get_current_page();
        $offset = HRM_Pagination::get_offset($current_page, $per_page);

        // Total records count
        $total_records = $wpdb->get_var( "SELECT COUNT(*) FROM $table" );

        // Fetch records with pagination
        $records = $wpdb->get_results( $wpdb->prepare(
            "SELECT * FROM $table ORDER BY date DESC, created_at DESC LIMIT %d OFFSET %d",
            $per_page, $offset
        ));
        
        // Identify Duplicates logic
        $counts = [];
        if ( $records ) {
            foreach ( $records as $r ) {
                $k = $r->employee_id . '_' . $r->date;
                if ( ! isset( $counts[$k] ) ) $counts[$k] = 0;
                $counts[$k]++; 
            }
        }
        ?>
        <div class="wrap hrm-wrap hrm-modern-dashboard">
            <div class="hrm-header-section">
                <div class="hrm-header-left">
                     <h1>Attendance</h1>
                     <p>Monitor employee check-ins and working hours.</p>
                </div>
            </div>

            <!-- Stats Cards -->
            <div class="hrm-stats-row">
               <div class="hrm-stat-card white">
                  <div class="stat-header">
                      <span>Checked In Today</span>
                      <span class="dashicons dashicons-yes-alt" style="color: #10B981;"></span>
                  </div>
                  <div class="stat-number"><?php echo $present_today; ?></div>
                  <div class="stat-footer">
                      <span class="badge-green">On Time</span>
                  </div>
               </div>
               <div class="hrm-stat-card white">
                  <div class="stat-header">
                      <span>Late Arrivals</span>
                      <span class="dashicons dashicons-warning" style="color: #F59E0B;"></span>
                  </div>
                  <div class="stat-number"><?php echo $late_today; ?></div>
                  <div class="stat-footer">
                      <span class="badge-yellow">Needs Attention</span>
                  </div>
               </div>
               <div class="hrm-stat-card white">
                  <div class="stat-header">
                      <span>Absent</span>
                      <span class="dashicons dashicons-dismiss" style="color: #EF4444;"></span>
                  </div>
                  <div class="stat-number"><?php echo $absent_today; ?></div>
                  <div class="stat-footer">
                      <span class="badge-red">Total Absent</span>
                  </div>
               </div>
               <div class="hrm-stat-card white">
                  <div class="stat-header">
                      <span>Total Records</span>
                      <span class="dashicons dashicons-database" style="color: #0C8776;"></span>
                  </div>
                  <div class="stat-number"><?php echo $total_today; ?></div>
                  <div class="stat-footer">
                      <span class="text-muted">For <?php echo date('M d'); ?></span>
                  </div>
               </div>
            </div>

            <!-- Filters & Table -->
            <div class="hrm-card" style="margin-top:24px; padding: 0; overflow: hidden; border: 1px solid #E5E7EB; border-radius: 12px; box-shadow: 0 1px 3px rgba(0,0,0,0.1);">
                <div class="hrm-filter-bar" style="border-bottom: 1px solid #F3F4F6; padding: 20px; background: white; flex-wrap: wrap;">
                    <div style="flex: 1; display: flex; gap: 12px;">
                        <input type="text" id="hrm-search-attendance" placeholder="Search employee..." class="hrm-search-input" style="max-width: 260px;">
                        <input type="date" id="hrm-date-filter" value="<?php echo date('Y-m-d'); ?>" class="hrm-select-input">
                        
                        <button class="hrm-btn-red" id="hrm-delete-bulk-btn" style="display: none;">
                            <span class="dashicons dashicons-trash"></span> Delete Selected
                        </button>
                    </div>
                    <div style="display:flex; gap: 10px;">
                         <button class="hrm-btn-primary" onclick="openModal('hrm-bulk-attendance-modal')" style="background: var(--hrm-gradient-primary);">
                            <span class="dashicons dashicons-groups" style="font-size: 16px; width: 16px; height: 16px; vertical-align: text-top;"></span> 
                            Bulk Attendance
                        </button>
                         <button class="hrm-btn-outline" onclick="openModal('hrm-manual-attendance-modal')">
                            <span class="dashicons dashicons-plus-alt2" style="font-size: 16px; width: 16px; height: 16px; vertical-align: text-top;"></span> 
                            Manual Check-in
                        </button>
                        <button class="hrm-btn-outline" onclick="window.location.reload()"><span class="dashicons dashicons-update"></span> Refresh</button>
                        <button class="hrm-btn-outline" id="hrm-export-btn" title="Export Month CSV">
                             <span class="dashicons dashicons-download"></span>
                        </button>
                    </div>
                </div>

                <div style="overflow-x: auto;">
                    <table class="hrm-table">
                      <thead>
                          <tr>
                              <th style="width: 40px; text-align: center;"><input type="checkbox" id="hrm-select-all-toggle"></th>
                              <th style="padding-left: 24px;">Employee</th>
                              <th>Date</th>
                              <th>Check In</th>
                              <th>Check Out</th>
                              <th>Hours</th>
                              <th>Status</th>
                              <th>Remarks</th>
                              <th style="text-align: right; padding-right: 24px;">Actions</th>
                          </tr>
                      </thead>
                      <tbody>
                          <?php if($records): foreach($records as $record): 
                               $employee = get_post($record->employee_id);
                               $employee_name = $employee ? $employee->post_title : 'Unknown';
                               $status_class = $record->status;
                               
                               // Format for inputs (H:i)
                               $in_time = $record->check_in ? date('H:i', strtotime($record->check_in)) : '';
                               $out_time = $record->check_out ? date('H:i', strtotime($record->check_out)) : '';
    
                               $k = $record->employee_id . '_' . $record->date;
                               $is_dup = isset($counts[$k]) && $counts[$k] > 1;
                               $row_style = $is_dup ? 'background-color: #FEF2F2;' : '';
                          ?>
                          <tr class="hrm-clickable-row <?php echo $is_dup ? 'hrm-duplicate-row' : ''; ?>" style="<?php echo $row_style; ?>">
                              <td style="text-align: center;">
                                  <input type="checkbox" class="hrm-row-checkbox" value="<?php echo $record->id; ?>">
                              </td>
                              <td style="padding-left: 24px;">
                                  <div class="hrm-user-cell">
                                      <div class="hrm-avatar-sm">
                                          <?php echo strtoupper(substr($employee_name, 0, 1)); ?>
                                      </div>
                                      <span style="font-weight: 600; color: #111827;"><?php echo esc_html($employee_name); ?>
                                      <?php if($is_dup): ?><span style="color:#EF4444; font-size:11px; margin-left:6px;">(Duplicate)</span><?php endif; ?>
                                      </span>
                                  </div>
                              </td>
                              <td>
                                  <span class="hrm-date-text">
                                    <?php echo date('M d, Y', strtotime($record->date)); ?>
                                  </span>
                              </td>
                              <td>
                                  <?php if($record->check_in): ?>
                                    <span class="hrm-time-pill">
                                        <?php echo date('h:i A', strtotime($record->check_in)); ?>
                                    </span>
                                  <?php else: ?>
                                    <span style="color:#D1D5DB">-</span>
                                  <?php endif; ?>
                              </td>
                              <td>
                                  <?php if($record->check_out): ?>
                                    <span class="hrm-time-pill">
                                        <?php echo date('h:i A', strtotime($record->check_out)); ?>
                                    </span>
                                  <?php else: ?>
                                    <span style="color:#D1D5DB">-</span>
                                  <?php endif; ?>
                              </td>
                              <td>
                                  <strong style="color: #374151;"><?php echo $record->hours_worked; ?></strong> <span style="color: #9CA3AF; font-size: 12px;">hrs</span>
                              </td>
                              <td>
                                  <?php 
                                    $badge_class = 'hrm-badge'; // Fallback
                                    if($status_class === 'present') $badge_class = 'badge-green';
                                    elseif($status_class === 'absent') $badge_class = 'badge-red';
                                    elseif($status_class === 'late') $badge_class = 'badge-yellow animated-late';
                                    elseif($status_class === 'early_leave') $badge_class = 'badge-yellow';
                                  ?>
                                  <span class="hrm-badge <?php echo $badge_class; ?>"><?php echo ucfirst(str_replace('_', ' ', $record->status)); ?></span>
                              </td>
                              <td>
                                  <?php 
                                    $rem = isset($record->remarks) ? $record->remarks : '';
                                    if($rem) {
                                        echo '<span title="'.esc_attr($rem).'" style="cursor:help; border-bottom:1px dotted #9CA3AF;">'.esc_html(mb_strimwidth($rem, 0, 20, '...')).'</span>';
                                    } else {
                                        echo '<span style="color:#D1D5DB">-</span>';
                                    }
                                  ?>
                              </td>
                              <td style="text-align: right; padding-right: 24px;">
                                  <button class="hrm-btn-icon hrm-edit-attendance-btn" 
                                          title="Edit Record"
                                          data-id="<?php echo $record->id; ?>"
                                          data-date="<?php echo $record->date; ?>"
                                          data-in="<?php echo $in_time; ?>"
                                          data-out="<?php echo $out_time; ?>"
                                          data-status="<?php echo $record->status; ?>"
                                          data-remarks="<?php echo isset($record->remarks) ? esc_attr($record->remarks) : ''; ?>">
                                      <span class="dashicons dashicons-edit"></span>
                                  </button>
                              </td>
                          </tr>
                          <?php endforeach; else: ?>
                          <tr><td colspan="8" style="text-align:center; padding: 60px; color: #9CA3AF;">No attendance records found for this period.</td></tr>
                          <?php endif; ?>
                      </tbody>
                    </table>
                </div>
                
                <!-- Pagination -->
                <?php 
                HRM_Pagination::render([
                    'total_items' => $total_records,
                    'current_page' => $current_page,
                    'per_page' => $per_page,
                    'show_sizes' => true,
                    'context' => 'admin'
                ]);
                HRM_Pagination::render_script();
                ?>
            </div>

            <!-- Edit Attendance Modal (Premium Design) -->
            <div id="hrm-edit-attendance-modal" class="hrm-modal" style="display:none;">
                <div class="hrm-modal-content hrm-premium-modal">
                    <div class="hrm-modal-header-premium">
                        <div class="header-icon-wrapper">
                            <span class="dashicons dashicons-edit"></span>
                        </div>
                        <div class="header-text">
                            <h2>Edit Attendance</h2>
                            <p>Update existing attendance record details.</p>
                        </div>
                        <button class="hrm-modal-close-premium" onclick="closeModal('hrm-edit-attendance-modal')">
                             <span class="dashicons dashicons-no-alt"></span>
                        </button>
                    </div>
                    
                    <form id="hrm-edit-attendance-form" class="hrm-premium-form-modal">
                        <input type="hidden" name="action" value="hrm_edit_attendance">
                        <input type="hidden" name="nonce" value="<?php echo wp_create_nonce( 'hrm_ajax_nonce' ); ?>">
                        <input type="hidden" name="record_id" id="edit_record_id">
                        
                        <div class="hrm-form-section">
                            <div class="hrm-form-group full-width">
                                <label>Date <span class="required">*</span></label>
                                <input type="date" name="date" id="edit_date" required>
                            </div>

                            <div class="hrm-row-modal">
                                <div class="hrm-form-group">
                                    <label>Check In Time</label>
                                    <input type="time" name="check_in" id="edit_check_in">
                                </div>
                                <div class="hrm-form-group">
                                    <label>Check Out Time</label>
                                    <input type="time" name="check_out" id="edit_check_out">
                                </div>
                            </div>

                            <div class="hrm-form-group full-width">
                                <label>Status</label>
                                <div class="hrm-select-wrapper">
                                    <select name="status" id="edit_status">
                                        <option value="present">Present</option>
                                        <option value="late">Late</option>
                                        <option value="absent">Absent</option>
                                        <option value="early_leave">Early Leave</option>
                                    </select>
                                    <span class="dashicons dashicons-arrow-down-alt2 select-arrow"></span>
                                </div>
                            </div>

                            <div class="hrm-form-group full-width">
                                <label>Remarks</label>
                                <textarea name="remarks" id="edit_remarks" rows="3" style="width:100%; padding: 10px; border: 1px solid #D1D5DB; border-radius: 6px;"></textarea>
                            </div>
                        </div>

                        <div class="hrm-modal-footer-premium">
                            <button type="button" class="hrm-btn-ghost" onclick="closeModal('hrm-edit-attendance-modal')">Cancel</button>
                            <button type="submit" class="hrm-btn-primary">Save Changes</button>
                        </div>
                    </form>
                </div>
            </div>

            <!-- Manual Attendance Modal (Premium Design) -->
            <div id="hrm-manual-attendance-modal" class="hrm-modal" style="display:none;">
                <div class="hrm-modal-content hrm-premium-modal">
                    <div class="hrm-modal-header-premium">
                        <div class="header-icon-wrapper">
                            <span class="dashicons dashicons-calendar-alt"></span>
                        </div>
                        <div class="header-text">
                            <h2>Manual Check-in</h2>
                            <p>Correct or add missing attendance records.</p>
                        </div>
                        <button class="hrm-modal-close-premium" onclick="closeModal('hrm-manual-attendance-modal')">
                            <span class="dashicons dashicons-no-alt"></span>
                        </button>
                    </div>
                    
                    <form id="hrm-manual-attendance-form" class="hrm-premium-form-modal">
                        <input type="hidden" name="action" value="hrm_manual_attendance">
                        <input type="hidden" name="nonce" value="<?php echo wp_create_nonce( 'hrm_ajax_nonce' ); ?>">
                        
                        <div class="hrm-form-section">
                            <div class="hrm-form-group full-width">
                                <label>Select Employee <span class="required">*</span></label>
                                <div class="hrm-select-wrapper">
                                    <select name="employee_id" required>
                                        <option value="">Choose an employee...</option>
                                        <?php 
                                        $employees = get_posts(['post_type' => HRM_Setup::EMPLOYEE_POST_TYPE, 'numberposts' => -1]);
                                        foreach($employees as $emp) {
                                            echo '<option value="'.$emp->ID.'">'.esc_html($emp->post_title).'</option>';
                                        }
                                        ?>
                                    </select>
                                    <span class="dashicons dashicons-arrow-down-alt2 select-arrow"></span>
                                </div>
                            </div>
                            
                            <div class="hrm-form-group full-width">
                                <label>Date <span class="required">*</span></label>
                                <input type="date" name="date" value="<?php echo date('Y-m-d'); ?>" required>
                            </div>

                            <div class="hrm-row-modal">
                                <div class="hrm-form-group">
                                    <label>Check In Time <span class="required">*</span></label>
                                    <input type="time" name="check_in" required>
                                </div>
                                <div class="hrm-form-group">
                                    <label>Check Out Time <span class="text-muted-xs">(Optional)</span></label>
                                    <input type="time" name="check_out">
                                </div>
                            </div>

                            <div class="hrm-form-group full-width">
                                <label>Remarks <span class="text-muted-xs">(Optional)</span></label>
                                <textarea name="remarks" rows="2" style="width:100%; padding: 10px; border: 1px solid #D1D5DB; border-radius: 6px;"></textarea>
                            </div>
                        </div>

                        <div class="hrm-modal-footer-premium">
                            <button type="button" class="hrm-btn-ghost" onclick="closeModal('hrm-manual-attendance-modal')">Cancel</button>
                            <button type="submit" class="hrm-btn-primary">Add Record</button>
                        </div>
                    </form>
                </div>
            <!-- Bulk Attendance Modal (New) -->
            <div id="hrm-bulk-attendance-modal" class="hrm-modal" style="display:none;">
                <div class="hrm-modal-content hrm-premium-modal" style="max-width: 600px;">
                    <div class="hrm-modal-header-premium">
                        <div class="header-icon-wrapper" style="background: var(--hrm-gradient-primary);">
                            <span class="dashicons dashicons-groups"></span>
                        </div>
                        <div class="header-text">
                            <h2>Bulk Attendance</h2>
                            <p>Mark attendance for multiple employees at once.</p>
                        </div>
                        <button class="hrm-modal-close-premium" onclick="closeModal('hrm-bulk-attendance-modal')">
                            <span class="dashicons dashicons-no-alt"></span>
                        </button>
                    </div>
                    
                    <form id="hrm-bulk-attendance-form" class="hrm-premium-form-modal">
                        <input type="hidden" name="action" value="hrm_bulk_attendance">
                        <input type="hidden" name="nonce" value="<?php echo wp_create_nonce( 'hrm_ajax_nonce' ); ?>">
                        
                        <div class="hrm-form-section">
                            <div class="hrm-form-group full-width">
                                <label>Select Employees <span class="required">*</span></label>
                                <div class="employee-selector-wrapper" style="border: 1px solid #D1D5DB; border-radius: 8px; padding: 12px; max-height: 200px; overflow-y: auto; background: #F9FAFB;">
                                    <div style="display: flex; align-items: center; gap: 8px; margin-bottom: 10px; padding-bottom: 10px; border-bottom: 1px solid #E5E7EB;">
                                        <input type="checkbox" id="bulk-select-all-emps">
                                        <label for="bulk-select-all-emps" style="font-weight: 700; cursor: pointer; margin: 0;">Select All Employees</label>
                                    </div>
                                    <div class="employee-checkbox-list" style="display: grid; grid-template-columns: 1fr 1fr; gap: 8px;">
                                        <?php 
                                        $all_employees = get_posts(['post_type' => HRM_Setup::EMPLOYEE_POST_TYPE, 'numberposts' => -1, 'post_status' => 'publish']);
                                        foreach($all_employees as $emp) {
                                            echo '<div style="display: flex; align-items: center; gap: 8px;">';
                                            echo '<input type="checkbox" name="employee_ids[]" value="'.$emp->ID.'" class="bulk-emp-checkbox" id="emp-'.$emp->ID.'">';
                                            echo '<label for="emp-'.$emp->ID.'" style="cursor: pointer; font-size: 13px; margin: 0;">'.esc_html($emp->post_title).'</label>';
                                            echo '</div>';
                                        }
                                        ?>
                                    </div>
                                </div>
                            </div>
                            
                            <div class="hrm-row-modal">
                                <div class="hrm-form-group">
                                    <label>Start Date <span class="required">*</span></label>
                                    <input type="date" name="start_date" value="<?php echo date('Y-m-d'); ?>" required>
                                </div>
                                <div class="hrm-form-group">
                                    <label>End Date <span class="required">*</span></label>
                                    <input type="date" name="end_date" value="<?php echo date('Y-m-d'); ?>" required>
                                </div>
                            </div>

                            <div class="hrm-form-group full-width">
                                <label>Attendance Status <span class="required">*</span></label>
                                <div class="hrm-select-wrapper">
                                    <select name="status" required>
                                        <option value="present">Present</option>
                                        <option value="absent">Absent</option>
                                        <option value="late">Late</option>
                                        <option value="half_day">Half Day</option>
                                        <option value="leave">Leave</option>
                                    </select>
                                    <span class="dashicons dashicons-arrow-down-alt2 select-arrow"></span>
                                </div>
                            </div>

                            <div class="hrm-form-group full-width">
                                <label>Remarks <span class="text-muted-xs">(Optional)</span></label>
                                <textarea name="bulk_remarks" rows="2" style="width:100%; padding: 10px; border: 1px solid #D1D5DB; border-radius: 6px;" placeholder="Add any common remarks for these entries..."></textarea>
                            </div>
                        </div>

                        <div class="hrm-modal-footer-premium">
                            <button type="button" class="hrm-btn-ghost" onclick="closeModal('hrm-bulk-attendance-modal')">Cancel</button>
                            <button type="submit" class="hrm-btn-primary" style="background: var(--hrm-gradient-primary);">Mark Bulk Attendance</button>
                        </div>
                    </form>
                </div>
            </div>
        </div>
        
        <script>
        jQuery(document).ready(function($) {
            
            // Global Helpers
            window.hrmToast = function(type, message) {
                var icon = type === 'success' ? 'dashicons-yes-alt' : 'dashicons-warning';
                var html = `
                <div class="hrm-toast-notification ${type}">
                    <span class="dashicons ${icon}"></span>
                    <span>${message}</span>
                    <button class="hrm-toast-close" onclick="this.parentElement.remove()">&times;</button>
                </div>`;
                
                $('body').append(html);
                setTimeout(function() {
                    $('.hrm-toast-notification').fadeOut(function() { $(this).remove(); });
                }, 4000);
            };

            window.openModal = function(id) {
                $('#' + id).css('display', 'flex').hide().fadeIn();
            };

            window.closeModal = function(id) {
                $('#' + id).fadeOut();
            };

            // Manual Attendance
            $('#hrm-manual-attendance-form').on('submit', function(e) {
                e.preventDefault();
                var btn = $(this).find('button[type="submit"]');
                var originalText = btn.text();
                btn.text('Adding...').prop('disabled', true);

                $.post(ajaxurl, $(this).serialize(), function(res) {
                    if(res.success) {
                        hrmToast('success', res.data);
                        setTimeout(function(){ location.reload(); }, 1000);
                    } else {
                        hrmToast('error', res.data);
                        btn.text(originalText).prop('disabled', false);
                    }
                });
            });

            // Edit Trigger
            $('.hrm-edit-attendance-btn').on('click', function() {
                var btn = $(this);
                $('#edit_record_id').val( btn.data('id') );
                $('#edit_date').val( btn.data('date') );
                $('#edit_check_in').val( btn.data('in') );
                $('#edit_check_out').val( btn.data('out') );
                $('#edit_status').val( btn.data('status') );
                $('#edit_remarks').val( btn.data('remarks') );
                
                openModal('hrm-edit-attendance-modal');
            });

            // Edit Submit
            $('#hrm-edit-attendance-form').on('submit', function(e) {
                e.preventDefault();
                var btn = $(this).find('button[type="submit"]');
                var originalText = btn.text();
                btn.text('Saving...').prop('disabled', true);

                $.post(ajaxurl, $(this).serialize(), function(res) {
                    if(res.success) {
                        hrmToast('success', res.data);
                        setTimeout(function(){ location.reload(); }, 1000);
                    } else {
                        hrmToast('error', res.data);
                        btn.text(originalText).prop('disabled', false);
                    }
                });
            });
            
            // Close Modals on Overlay Click
            $(window).on('click', function(e) {
                if ($(e.target).is('.hrm-modal')) {
                    $('.hrm-modal').fadeOut();
                }
            });

            // Export
            $('#hrm-export-btn').on('click', function() {
                var date = $('#hrm-date-filter').val();
                window.location.href = ajaxurl + '?action=hrm_export_attendance&date=' + date + '&nonce=' + '<?php echo wp_create_nonce("hrm_ajax_nonce"); ?>';
            });
        });
        </script>
        <?php
    }

    public static function render_frontend_attendance( $employee_id ) {
        global $wpdb;
        $today = current_time( 'Y-m-d' );
        
        $record = $wpdb->get_row( $wpdb->prepare( 
            "SELECT * FROM {$wpdb->prefix}hrm_attendance WHERE employee_id = %d AND date = %s", 
            $employee_id, $today 
        ));
    
        $check_in = $record ? $record->check_in : null;
        $check_out = $record ? $record->check_out : null;
        $hours_worked = $record && isset($record->hours_worked) ? $record->hours_worked : 0;
        
        $visual_status = 'Pending';
        $status_class = 'pending';
        $timer_active_class = '';
    
        $current_ts_raw = time(); // Standard UTC timestamp
        $check_in_gmt = $check_in ? get_gmt_from_date( $check_in ) : null;
        $check_out_gmt = $check_out ? get_gmt_from_date( $check_out ) : null;
        
        $check_in_ts = $check_in_gmt ? strtotime( $check_in_gmt ) : 0;
        $check_out_ts = $check_out_gmt ? strtotime( $check_out_gmt ) : 0;
    
        $is_active_session = ( $check_in && ! $check_out );
    
        if ( $is_active_session ) {
            $visual_status = 'Checked In';
            $status_class = 'active'; 
            $timer_active_class = 'active';
        } elseif ( $check_in && $check_out ) {
            $visual_status = 'Checked Out';
            $status_class = 'completed'; 
        } elseif ( $record && $record->status === 'absent' && ! $check_in ) {
            $visual_status = 'Absent';
            $status_class = 'error'; 
        }
    
        // Defined above via UTC conversion
        
        $history = $wpdb->get_results( $wpdb->prepare( 
            "SELECT * FROM {$wpdb->prefix}hrm_attendance WHERE employee_id = %d ORDER BY date DESC LIMIT 7", 
            $employee_id 
        ));
        ?>
        <div class="hrm-attendance-box hrm-attendance-dashboard">
            <div class="hrm-attendance-hero">
                <div class="hrm-status-indicator <?php echo $status_class; ?>">
                    <span class="status-dot"></span>
                    <?php echo $visual_status; ?>
                </div>
    
                <div class="hrm-timer-wrapper <?php echo $timer_active_class; ?>">
                    <div class="hrm-timer-content">
                        <?php 
                            $elapsed = 0;
                            if ( $is_active_session && $check_in_ts > 0 ) {
                                // Use raw PHP time() for consistency with strtotime()
                                // Both return UNIX timestamps in the same timezone
                                $elapsed = $current_ts_raw - $check_in_ts;
                            } elseif ( $check_in && $check_out ) {
                                // Both strtotime() results - consistent
                                $elapsed = $check_out_ts - $check_in_ts;
                            }
                            // Extra safety: ensure non-negative
                            if ($elapsed < 0) $elapsed = 0;
                        ?>
                        <div id="hrm-hero-timer" class="hrm-timer-bg" data-elapsed="<?php echo intval($elapsed); ?>" data-active="<?php echo $is_active_session ? '1' : '0'; ?>">
                            <?php echo ($check_out || !$check_in || $elapsed == 0) ? '00:00:00' : gmdate('H:i:s', $elapsed); ?>
                        </div>
                        <span class="hrm-timer-label">
                            <?php echo $check_out ? 'Total Hours' : 'Duration'; ?>
                        </span>
                    </div>
                </div>
    
                <?php if ( $hours_worked > 0 && ! $is_active_session ) : ?>
                    <div style="font-size: 18px; font-weight: 600; color: var(--text-main); margin-bottom: 20px;">
                        You worked <span style="color: var(--primary-color);"><?php echo $hours_worked; ?> hrs</span> today.
                    </div>
                <?php endif; ?>
    
                <form method="post" class="hrm-action-area" id="hrm-attendance-form">
                    <?php wp_nonce_field( 'hrm_attendance_action', 'hrm_attendance_nonce' ); ?>
                    <!-- AJAX Nonce for JavaScript fallback -->
                    <input type="hidden" name="hrm_ajax_nonce" id="hrm_ajax_nonce_field" value="<?php echo wp_create_nonce('hrm_ajax_nonce'); ?>">
                    <input type="hidden" name="hrm_action" value="mark_attendance">
                    <input type="hidden" name="employee_id" value="<?php echo $employee_id; ?>">
                    
                <?php if ( ! $check_in ) : ?>
                        <button type="submit" id="hrm-btn-checkin" name="type" value="check_in" class="hrm-btn hrm-btn-large hrm-btn-green pulse hrm-check-in-btn">
                            <span class="dashicons dashicons-controls-play" style="font-size: 24px; width: 24px; height: 24px;"></span>
                            Check In
                        </button>
                    <?php elseif ( $is_active_session ) : ?>
                        <button type="submit" id="hrm-btn-checkout" name="type" value="check_out" class="hrm-btn hrm-btn-large hrm-btn-red">
                            <span class="dashicons dashicons-controls-pause" style="font-size: 24px; width: 24px; height: 24px;"></span>
                            Check Out
                        </button>

                    <?php else : ?>
                        <div style="color: var(--text-secondary); font-size: 14px; margin-top: -10px;">
                            <p><?php echo ($record && $record->status === 'absent') ? 'Marked as Absent for today.' : 'Have a great rest of your day!'; ?></p>
                        </div>
                    <?php endif; ?>
                </form>
            </div>
    
            <?php if ( $history ) : ?>
            <div class="hrm-history-section">
                <h4>Recent Activity</h4>
                <table class="hrm-history-table">
                    <thead>
                        <tr>
                            <th>Date</th>
                            <th>Check In</th>
                            <th>Check Out</th>
                            <th>Hours</th>
                            <th>Status</th>
                        </tr>
                    </thead>
                    <tbody>
                        <?php foreach ( $history as $log ) :
                            $status_badge_class = 'pending';
                            if ($log->status === 'present') $status_badge_class = 'success';
                            elseif ($log->status === 'absent') $status_badge_class = 'error';
                            elseif ($log->status === 'late') $status_badge_class = 'warning';
                        ?>
                        <tr>
                            <td><?php echo date('M d, Y', strtotime($log->date)); ?></td>
                            <td><?php echo $log->check_in ? date('h:i A', strtotime($log->check_in)) : '-'; ?></td>
                            <td><?php echo $log->check_out ? date('h:i A', strtotime($log->check_out)) : '-'; ?></td>
                            <td style="font-weight: 600;"><?php echo $log->hours_worked ? $log->hours_worked . ' hrs' : '-'; ?></td>
                            <td>
                                <span class="hrm-notice <?php echo $status_badge_class . ($log->status === 'late' ? ' animated-late' : ''); ?>" style="padding: 2px 8px; font-size: 11px; margin: 0;">
                                    <?php echo ucfirst(str_replace('_', ' ', $log->status)); ?>
                                </span>
                            </td>
                        </tr>
                        <?php endforeach; ?>
                    </tbody>
                </table>
            </div>
            <?php endif; ?>
        </div>
        <?php
    }

    public static function handle_frontend_attendance( $employee_id ) {
        if ( ! isset( $_POST['hrm_action'] ) || $_POST['hrm_action'] !== 'mark_attendance' ) return;
        check_admin_referer( 'hrm_attendance_action' );

        global $wpdb;
        $type = sanitize_key( $_POST['type'] );
        $today = current_time( 'Y-m-d' );
        $now = current_time( 'mysql' );

        if ( $type === 'check_in' ) {
            // Get ALL records for today to detect duplicates (Robustness)
            $all_records = $wpdb->get_results( $wpdb->prepare(
                "SELECT id, status, check_in FROM {$wpdb->prefix}hrm_attendance WHERE employee_id = %d AND date = %s ORDER BY id ASC",
                $employee_id, $today
            ));

            $existing_record = null;

            if ( ! empty( $all_records ) ) {
                $existing_record = $all_records[0];
                
                // Cleanup Duplicates
                if ( count( $all_records ) > 1 ) {
                     $ids_to_delete = [];
                     for ( $i = 1; $i < count( $all_records ); $i++ ) {
                         $ids_to_delete[] = $all_records[$i]->id;
                     }
                     if ( ! empty( $ids_to_delete ) ) {
                         $ids_str = implode(',', array_map('intval', $ids_to_delete));
                         $wpdb->query( "DELETE FROM {$wpdb->prefix}hrm_attendance WHERE id IN ($ids_str)" );
                     }
                }
            }

            if ( $existing_record && $existing_record->status === 'absent' ) {
                 // Locked
                 wp_die('<h1>Check-in Failed</h1><p>You have been marked as <strong>Absent</strong> for today. Please contact admin.</p><a href="'.remove_query_arg('hrm_action').'">Back</a>');
            }

            // Strict Rule: Employee can check in only once per day.
            if ( $existing_record && ! empty( $existing_record->check_in ) ) {
                 wp_redirect( remove_query_arg('hrm_action') );
                 exit;
            }

            // Always Present on Check-In
            $status = 'present';

            // Set Check-In
            if ( $existing_record ) {
                $wpdb->update( $wpdb->prefix . 'hrm_attendance', [
                    'check_in'    => $now,
                    'status'      => $status
                ], [ 'id' => $existing_record->id ] );
            } else {
                $wpdb->insert( $wpdb->prefix . 'hrm_attendance', [
                    'employee_id' => $employee_id,
                    'date' => $today,
                    'check_in' => $now,
                    'check_out' => null, 
                    'hours_worked' => 0, 
                    'status' => $status
                ]);
            }
        } elseif ( $type === 'check_out' ) {
            // Calculate Hours Worked
            $check_in_time = $wpdb->get_var( $wpdb->prepare(
                "SELECT check_in FROM {$wpdb->prefix}hrm_attendance WHERE employee_id = %d AND date = %s",
                $employee_id, $today
            ));

            $hours_worked = 0;
            if ( $check_in_time ) {
                $start = strtotime( get_gmt_from_date( $check_in_time ) );
                $end = time();
                $diff = $end - $start;
                $hours_worked = round( $diff / 3600, 2 ); 
            }

            $wpdb->update( $wpdb->prefix . 'hrm_attendance', 
                [ 
                    'check_out' => $now,
                    'status' => 'present',
                    'hours_worked' => $hours_worked
                ],
                [ 'employee_id' => $employee_id, 'date' => $today ]
            );
        }
        
        // Redirect to avoid resubmit
        wp_redirect( remove_query_arg('hrm_action') ); 
        exit;
    }
}
